/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.util;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.DataInputStream;
import java.io.DataOutput;
import java.io.DataOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Set;
import java.util.TreeSet;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import org.apache.hadoop.hbase.HConstants;
import org.apache.hadoop.hbase.nio.ByteBuff;
import org.apache.hadoop.hbase.util.ByteBufferUtils;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.io.WritableUtils;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ByteBufferUtilsTestBase {
    private static final Logger LOG = LoggerFactory.getLogger(ByteBufferUtilsTestBase.class);
    private static int MAX_VLONG_LENGTH = 9;
    private static Collection<Long> testNumbers;
    private byte[] array;

    @BeforeAll
    public static void setUpBeforeAll() {
        TreeSet<Long> a = new TreeSet<Long>();
        for (int i = 0; i <= 63; ++i) {
            long v = -1L << i;
            Assertions.assertTrue((v < 0L ? 1 : 0) != 0);
            ByteBufferUtilsTestBase.addNumber(a, v);
            v = (1L << i) - 1L;
            Assertions.assertTrue((v >= 0L ? 1 : 0) != 0);
            ByteBufferUtilsTestBase.addNumber(a, v);
        }
        testNumbers = Collections.unmodifiableSet(a);
        LOG.info("Testing variable-length long serialization using: {} (count: {})", testNumbers, (Object)testNumbers.size());
        Assertions.assertEquals((int)1753, (int)testNumbers.size());
        Assertions.assertEquals((long)Long.MIN_VALUE, (long)((Long)a.first()));
        Assertions.assertEquals((long)Long.MAX_VALUE, (long)((Long)a.last()));
    }

    @BeforeEach
    public void setUp() {
        this.array = new byte[8];
        for (int i = 0; i < this.array.length; ++i) {
            this.array[i] = (byte)(97 + i);
        }
    }

    private static void addNumber(Set<Long> a, long l) {
        if (l != Long.MIN_VALUE) {
            a.add(l - 1L);
        }
        a.add(l);
        if (l != Long.MAX_VALUE) {
            a.add(l + 1L);
        }
        for (long divisor = 3L; divisor <= 10L; ++divisor) {
            for (long delta = -1L; delta <= 1L; ++delta) {
                a.add(l / divisor + delta);
            }
        }
    }

    @Test
    public void testReadWriteVLong() {
        for (long l : testNumbers) {
            ByteBuffer b = ByteBuffer.allocate(MAX_VLONG_LENGTH);
            ByteBufferUtils.writeVLong((ByteBuffer)b, (long)l);
            b.flip();
            Assertions.assertEquals((long)l, (long)ByteBufferUtils.readVLong((ByteBuffer)b));
            b.flip();
            Assertions.assertEquals((long)l, (long)ByteBufferUtils.readVLong((ByteBuff)ByteBuff.wrap((ByteBuffer)b)));
        }
    }

    @Test
    public void testReadWriteConsecutiveVLong() {
        for (long l : testNumbers) {
            ByteBuffer b = ByteBuffer.allocate(2 * MAX_VLONG_LENGTH);
            ByteBufferUtils.writeVLong((ByteBuffer)b, (long)l);
            ByteBufferUtils.writeVLong((ByteBuffer)b, (long)(l - 4L));
            b.flip();
            Assertions.assertEquals((long)l, (long)ByteBufferUtils.readVLong((ByteBuffer)b));
            Assertions.assertEquals((long)(l - 4L), (long)ByteBufferUtils.readVLong((ByteBuffer)b));
            b.flip();
            Assertions.assertEquals((long)l, (long)ByteBufferUtils.readVLong((ByteBuff)ByteBuff.wrap((ByteBuffer)b)));
            Assertions.assertEquals((long)(l - 4L), (long)ByteBufferUtils.readVLong((ByteBuff)ByteBuff.wrap((ByteBuffer)b)));
        }
    }

    @Test
    public void testConsistencyWithHadoopVLong() throws IOException {
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        DataOutputStream dos = new DataOutputStream(baos);
        for (long l : testNumbers) {
            baos.reset();
            ByteBuffer b = ByteBuffer.allocate(MAX_VLONG_LENGTH);
            ByteBufferUtils.writeVLong((ByteBuffer)b, (long)l);
            String bufStr = Bytes.toStringBinary((byte[])b.array(), (int)b.arrayOffset(), (int)b.position());
            WritableUtils.writeVLong((DataOutput)dos, (long)l);
            String baosStr = Bytes.toStringBinary((byte[])baos.toByteArray());
            Assertions.assertEquals((Object)baosStr, (Object)bufStr);
        }
    }

    @Test
    public void testMoveBufferToStream() throws IOException {
        int arrayOffset = 7;
        int initialPosition = 10;
        int endPadding = 5;
        byte[] arrayWrapper = new byte[17 + this.array.length + 5];
        System.arraycopy(this.array, 0, arrayWrapper, 17, this.array.length);
        ByteBuffer buffer = ByteBuffer.wrap(arrayWrapper, 7, 10 + this.array.length).slice();
        Assertions.assertEquals((int)(10 + this.array.length), (int)buffer.limit());
        Assertions.assertEquals((int)0, (int)buffer.position());
        buffer.position(10);
        ByteArrayOutputStream bos = new ByteArrayOutputStream();
        ByteBufferUtils.moveBufferToStream((OutputStream)bos, (ByteBuffer)buffer, (int)this.array.length);
        Assertions.assertArrayEquals((byte[])this.array, (byte[])bos.toByteArray());
        Assertions.assertEquals((int)(10 + this.array.length), (int)buffer.position());
    }

    @Test
    public void testCopyToStreamWithOffset() throws IOException {
        ByteBuffer buffer = ByteBuffer.wrap(this.array);
        ByteArrayOutputStream bos = new ByteArrayOutputStream();
        ByteBufferUtils.copyBufferToStream((OutputStream)bos, (ByteBuffer)buffer, (int)(this.array.length / 2), (int)(this.array.length / 2));
        byte[] returnedArray = bos.toByteArray();
        for (int i = 0; i < this.array.length / 2; ++i) {
            int pos = this.array.length / 2 + i;
            Assertions.assertEquals((byte)returnedArray[i], (byte)this.array[pos]);
        }
    }

    @Test
    public void testCopyFromStream() throws IOException {
        ByteBuffer buffer = ByteBuffer.allocate(this.array.length);
        ByteArrayInputStream bis = new ByteArrayInputStream(this.array);
        DataInputStream dis = new DataInputStream(bis);
        ByteBufferUtils.copyFromStreamToBuffer((ByteBuffer)buffer, (DataInputStream)dis, (int)(this.array.length / 2));
        ByteBufferUtils.copyFromStreamToBuffer((ByteBuffer)buffer, (DataInputStream)dis, (int)(this.array.length - this.array.length / 2));
        for (int i = 0; i < this.array.length; ++i) {
            Assertions.assertEquals((byte)this.array[i], (byte)buffer.get(i));
        }
    }

    @Test
    public void testCopyFromBuffer() {
        ByteBuffer srcBuffer = ByteBuffer.allocate(this.array.length);
        ByteBuffer dstBuffer = ByteBuffer.allocate(this.array.length);
        srcBuffer.put(this.array);
        ByteBufferUtils.copyFromBufferToBuffer((ByteBuffer)srcBuffer, (ByteBuffer)dstBuffer, (int)(this.array.length / 2), (int)(this.array.length / 4));
        for (int i = 0; i < this.array.length / 4; ++i) {
            Assertions.assertEquals((byte)srcBuffer.get(i + this.array.length / 2), (byte)dstBuffer.get(i));
        }
    }

    @Test
    public void testCompressedInt() throws IOException {
        int i;
        ByteBufferUtilsTestBase.testCompressedInt(0);
        ByteBufferUtilsTestBase.testCompressedInt(Integer.MAX_VALUE);
        ByteBufferUtilsTestBase.testCompressedInt(Integer.MIN_VALUE);
        for (i = 0; i < 3; ++i) {
            ByteBufferUtilsTestBase.testCompressedInt((128 << i) - 1);
        }
        for (i = 0; i < 3; ++i) {
            ByteBufferUtilsTestBase.testCompressedInt(128 << i);
        }
    }

    @Test
    public void testIntFitsIn() {
        Assertions.assertEquals((int)1, (int)ByteBufferUtils.intFitsIn((int)0));
        Assertions.assertEquals((int)1, (int)ByteBufferUtils.intFitsIn((int)1));
        Assertions.assertEquals((int)2, (int)ByteBufferUtils.intFitsIn((int)256));
        Assertions.assertEquals((int)3, (int)ByteBufferUtils.intFitsIn((int)65536));
        Assertions.assertEquals((int)4, (int)ByteBufferUtils.intFitsIn((int)-1));
        Assertions.assertEquals((int)4, (int)ByteBufferUtils.intFitsIn((int)Integer.MAX_VALUE));
        Assertions.assertEquals((int)4, (int)ByteBufferUtils.intFitsIn((int)Integer.MIN_VALUE));
    }

    @Test
    public void testLongFitsIn() {
        Assertions.assertEquals((int)1, (int)ByteBufferUtils.longFitsIn((long)0L));
        Assertions.assertEquals((int)1, (int)ByteBufferUtils.longFitsIn((long)1L));
        Assertions.assertEquals((int)3, (int)ByteBufferUtils.longFitsIn((long)65536L));
        Assertions.assertEquals((int)5, (int)ByteBufferUtils.longFitsIn((long)0x100000000L));
        Assertions.assertEquals((int)8, (int)ByteBufferUtils.longFitsIn((long)-1L));
        Assertions.assertEquals((int)8, (int)ByteBufferUtils.longFitsIn((long)Long.MIN_VALUE));
        Assertions.assertEquals((int)8, (int)ByteBufferUtils.longFitsIn((long)Long.MAX_VALUE));
    }

    @Test
    public void testArePartEqual() {
        byte[] array = new byte[]{1, 2, 3, 4, 5, 1, 2, 3, 4};
        ByteBuffer buffer = ByteBuffer.wrap(array);
        Assertions.assertTrue((boolean)ByteBufferUtils.arePartsEqual((ByteBuffer)buffer, (int)0, (int)4, (int)5, (int)4));
        Assertions.assertTrue((boolean)ByteBufferUtils.arePartsEqual((ByteBuffer)buffer, (int)1, (int)2, (int)6, (int)2));
        Assertions.assertFalse((boolean)ByteBufferUtils.arePartsEqual((ByteBuffer)buffer, (int)1, (int)2, (int)6, (int)3));
        Assertions.assertFalse((boolean)ByteBufferUtils.arePartsEqual((ByteBuffer)buffer, (int)1, (int)3, (int)6, (int)2));
        Assertions.assertFalse((boolean)ByteBufferUtils.arePartsEqual((ByteBuffer)buffer, (int)0, (int)3, (int)6, (int)3));
    }

    @Test
    public void testPutInt() {
        int i;
        ByteBufferUtilsTestBase.testPutInt(0);
        ByteBufferUtilsTestBase.testPutInt(Integer.MAX_VALUE);
        for (i = 0; i < 3; ++i) {
            ByteBufferUtilsTestBase.testPutInt((128 << i) - 1);
        }
        for (i = 0; i < 3; ++i) {
            ByteBufferUtilsTestBase.testPutInt(128 << i);
        }
    }

    @Test
    public void testToBytes() {
        ByteBuffer buffer = ByteBuffer.allocate(5);
        buffer.put(new byte[]{0, 1, 2, 3, 4});
        Assertions.assertEquals((int)5, (int)buffer.position());
        Assertions.assertEquals((int)5, (int)buffer.limit());
        byte[] copy = ByteBufferUtils.toBytes((ByteBuffer)buffer, (int)2);
        Assertions.assertArrayEquals((byte[])new byte[]{2, 3, 4}, (byte[])copy);
        Assertions.assertEquals((int)5, (int)buffer.position());
        Assertions.assertEquals((int)5, (int)buffer.limit());
    }

    @Test
    public void testToPrimitiveTypes() {
        ByteBuffer buffer = ByteBuffer.allocate(15);
        long l = 988L;
        int i = 135;
        short s = 7;
        buffer.putLong(l);
        buffer.putShort(s);
        buffer.putInt(i);
        Assertions.assertEquals((long)l, (long)ByteBufferUtils.toLong((ByteBuffer)buffer, (int)0));
        Assertions.assertEquals((short)s, (short)ByteBufferUtils.toShort((ByteBuffer)buffer, (int)8));
        Assertions.assertEquals((int)i, (int)ByteBufferUtils.toInt((ByteBuffer)buffer, (int)10));
    }

    @Test
    public void testCopyFromArrayToBuffer() {
        byte[] b = new byte[15];
        b[0] = -1;
        long l = 988L;
        int i = 135;
        short s = 7;
        Bytes.putLong((byte[])b, (int)1, (long)l);
        Bytes.putShort((byte[])b, (int)9, (short)s);
        Bytes.putInt((byte[])b, (int)11, (int)i);
        ByteBuffer buffer = ByteBuffer.allocate(14);
        ByteBufferUtils.copyFromArrayToBuffer((ByteBuffer)buffer, (byte[])b, (int)1, (int)14);
        buffer.rewind();
        Assertions.assertEquals((long)l, (long)buffer.getLong());
        Assertions.assertEquals((short)s, (short)buffer.getShort());
        Assertions.assertEquals((int)i, (int)buffer.getInt());
    }

    private void testCopyFromSrcToDestWithThreads(Object input, Object output, List<Integer> lengthes, List<Integer> offsets) throws InterruptedException {
        Assertions.assertTrue((input instanceof ByteBuffer || input instanceof byte[] ? 1 : 0) != 0);
        Assertions.assertTrue((output instanceof ByteBuffer || output instanceof byte[] ? 1 : 0) != 0);
        Assertions.assertEquals((int)lengthes.size(), (int)offsets.size());
        int threads = lengthes.size();
        CountDownLatch latch = new CountDownLatch(1);
        ArrayList<Runnable> exes = new ArrayList<Runnable>(threads);
        int oldInputPos = input instanceof ByteBuffer ? ((ByteBuffer)input).position() : 0;
        int oldOutputPos = output instanceof ByteBuffer ? ((ByteBuffer)output).position() : 0;
        for (int i = 0; i != threads; ++i) {
            int offset = offsets.get(i);
            int length = lengthes.get(i);
            exes.add(() -> {
                try {
                    latch.await();
                    if (input instanceof ByteBuffer && output instanceof byte[]) {
                        ByteBufferUtils.copyFromBufferToArray((byte[])((byte[])output), (ByteBuffer)((ByteBuffer)input), (int)offset, (int)offset, (int)length);
                    }
                    if (input instanceof byte[] && output instanceof ByteBuffer) {
                        ByteBufferUtils.copyFromArrayToBuffer((ByteBuffer)((ByteBuffer)output), (int)offset, (byte[])((byte[])input), (int)offset, (int)length);
                    }
                    if (input instanceof ByteBuffer && output instanceof ByteBuffer) {
                        ByteBufferUtils.copyFromBufferToBuffer((ByteBuffer)((ByteBuffer)input), (ByteBuffer)((ByteBuffer)output), (int)offset, (int)offset, (int)length);
                    }
                }
                catch (InterruptedException ex) {
                    throw new RuntimeException(ex);
                }
            });
        }
        ExecutorService service = Executors.newFixedThreadPool(threads);
        exes.forEach(service::execute);
        latch.countDown();
        service.shutdown();
        Assertions.assertTrue((boolean)service.awaitTermination(5L, TimeUnit.SECONDS));
        if (input instanceof ByteBuffer) {
            Assertions.assertEquals((int)oldInputPos, (int)((ByteBuffer)input).position());
        }
        if (output instanceof ByteBuffer) {
            Assertions.assertEquals((int)oldOutputPos, (int)((ByteBuffer)output).position());
        }
        String inputString = input instanceof ByteBuffer ? Bytes.toString((byte[])Bytes.toBytes((ByteBuffer)((ByteBuffer)input))) : Bytes.toString((byte[])((byte[])input));
        String outputString = output instanceof ByteBuffer ? Bytes.toString((byte[])Bytes.toBytes((ByteBuffer)((ByteBuffer)output))) : Bytes.toString((byte[])((byte[])output));
        Assertions.assertEquals((Object)inputString, (Object)outputString);
    }

    @Test
    public void testCopyFromSrcToDestWithThreads() throws InterruptedException {
        List<byte[]> words = Arrays.asList(Bytes.toBytes((String)"with"), Bytes.toBytes((String)"great"), Bytes.toBytes((String)"power"), Bytes.toBytes((String)"comes"), Bytes.toBytes((String)"great"), Bytes.toBytes((String)"responsibility"));
        List<Integer> lengthes = words.stream().map(v -> ((byte[])v).length).collect(Collectors.toList());
        ArrayList<Integer> offsets = new ArrayList<Integer>(words.size());
        for (int i = 0; i != words.size(); ++i) {
            offsets.add(words.subList(0, i).stream().mapToInt(v -> ((byte[])v).length).sum());
        }
        int totalSize = words.stream().mapToInt(v -> ((byte[])v).length).sum();
        byte[] fullContent = new byte[totalSize];
        int offset = 0;
        for (byte[] w : words) {
            offset = Bytes.putBytes((byte[])fullContent, (int)offset, (byte[])w, (int)0, (int)w.length);
        }
        for (ByteBuffer input : Arrays.asList(ByteBuffer.allocateDirect(totalSize), ByteBuffer.allocate(totalSize))) {
            words.forEach(input::put);
            byte[] output = new byte[totalSize];
            this.testCopyFromSrcToDestWithThreads(input, output, lengthes, offsets);
        }
        for (ByteBuffer output : Arrays.asList(ByteBuffer.allocateDirect(totalSize), ByteBuffer.allocate(totalSize))) {
            byte[] input = fullContent;
            this.testCopyFromSrcToDestWithThreads(input, output, lengthes, offsets);
        }
        for (ByteBuffer input : Arrays.asList(ByteBuffer.allocateDirect(totalSize), ByteBuffer.allocate(totalSize))) {
            words.forEach(input::put);
            for (ByteBuffer output : Arrays.asList(ByteBuffer.allocateDirect(totalSize), ByteBuffer.allocate(totalSize))) {
                this.testCopyFromSrcToDestWithThreads(input, output, lengthes, offsets);
            }
        }
    }

    @Test
    public void testCopyFromBufferToArray() {
        ByteBuffer buffer = ByteBuffer.allocate(15);
        buffer.put((byte)-1);
        long l = 988L;
        int i = 135;
        short s = 7;
        buffer.putShort(s);
        buffer.putInt(i);
        buffer.putLong(l);
        byte[] b = new byte[15];
        ByteBufferUtils.copyFromBufferToArray((byte[])b, (ByteBuffer)buffer, (int)1, (int)1, (int)14);
        Assertions.assertEquals((short)s, (short)Bytes.toShort((byte[])b, (int)1));
        Assertions.assertEquals((int)i, (int)Bytes.toInt((byte[])b, (int)3));
        Assertions.assertEquals((long)l, (long)Bytes.toLong((byte[])b, (int)7));
    }

    @Test
    public void testRelativeCopyFromBuffertoBuffer() {
        ByteBuffer bb1 = ByteBuffer.allocate(135);
        ByteBuffer bb2 = ByteBuffer.allocate(135);
        ByteBufferUtilsTestBase.fillBB(bb1, (byte)5);
        ByteBufferUtils.copyFromBufferToBuffer((ByteBuffer)bb1, (ByteBuffer)bb2);
        Assertions.assertTrue((bb1.position() == bb2.position() ? 1 : 0) != 0);
        Assertions.assertTrue((bb1.limit() == bb2.limit() ? 1 : 0) != 0);
        bb1 = ByteBuffer.allocateDirect(135);
        bb2 = ByteBuffer.allocateDirect(135);
        ByteBufferUtilsTestBase.fillBB(bb1, (byte)5);
        ByteBufferUtils.copyFromBufferToBuffer((ByteBuffer)bb1, (ByteBuffer)bb2);
        Assertions.assertTrue((bb1.position() == bb2.position() ? 1 : 0) != 0);
        Assertions.assertTrue((bb1.limit() == bb2.limit() ? 1 : 0) != 0);
    }

    @Test
    public void testCompareTo() {
        ByteBuffer bb1 = ByteBuffer.allocate(135);
        ByteBuffer bb2 = ByteBuffer.allocate(135);
        byte[] b = new byte[71];
        ByteBufferUtilsTestBase.fillBB(bb1, (byte)5);
        ByteBufferUtilsTestBase.fillBB(bb2, (byte)5);
        ByteBufferUtilsTestBase.fillArray(b, (byte)5);
        Assertions.assertEquals((int)0, (int)ByteBufferUtils.compareTo((ByteBuffer)bb1, (int)0, (int)bb1.remaining(), (ByteBuffer)bb2, (int)0, (int)bb2.remaining()));
        Assertions.assertTrue((ByteBufferUtils.compareTo((ByteBuffer)bb1, (int)0, (int)bb1.remaining(), (byte[])b, (int)0, (int)b.length) > 0 ? 1 : 0) != 0);
        bb2.put(134, (byte)6);
        Assertions.assertTrue((ByteBufferUtils.compareTo((ByteBuffer)bb1, (int)0, (int)bb1.remaining(), (ByteBuffer)bb2, (int)0, (int)bb2.remaining()) < 0 ? 1 : 0) != 0);
        bb2.put(6, (byte)4);
        Assertions.assertTrue((ByteBufferUtils.compareTo((ByteBuffer)bb1, (int)0, (int)bb1.remaining(), (ByteBuffer)bb2, (int)0, (int)bb2.remaining()) > 0 ? 1 : 0) != 0);
        ByteBuffer bb3 = ByteBuffer.allocate(135);
        ByteBufferUtilsTestBase.fillBB(bb3, (byte)0);
        byte[] b3 = new byte[135];
        ByteBufferUtilsTestBase.fillArray(b3, (byte)1);
        int result = ByteBufferUtils.compareTo((byte[])b3, (int)0, (int)b3.length, (ByteBuffer)bb3, (int)0, (int)bb3.remaining());
        Assertions.assertTrue((result > 0 ? 1 : 0) != 0);
        result = ByteBufferUtils.compareTo((ByteBuffer)bb3, (int)0, (int)bb3.remaining(), (byte[])b3, (int)0, (int)b3.length);
        Assertions.assertTrue((result < 0 ? 1 : 0) != 0);
        byte[] b4 = Bytes.toBytes((String)"123");
        ByteBuffer bb4 = ByteBuffer.allocate(10 + b4.length);
        for (int i = 10; i < bb4.capacity(); ++i) {
            bb4.put(i, b4[i - 10]);
        }
        result = ByteBufferUtils.compareTo((byte[])b4, (int)0, (int)b4.length, (ByteBuffer)bb4, (int)10, (int)b4.length);
        Assertions.assertEquals((int)0, (int)result);
    }

    @Test
    public void testEquals() {
        byte[] a = Bytes.toBytes((String)"http://A");
        ByteBuffer bb = ByteBuffer.wrap(a);
        Assertions.assertTrue((boolean)ByteBufferUtils.equals((ByteBuffer)HConstants.EMPTY_BYTE_BUFFER, (int)0, (int)0, (ByteBuffer)HConstants.EMPTY_BYTE_BUFFER, (int)0, (int)0));
        Assertions.assertFalse((boolean)ByteBufferUtils.equals((ByteBuffer)HConstants.EMPTY_BYTE_BUFFER, (int)0, (int)0, (ByteBuffer)bb, (int)0, (int)a.length));
        Assertions.assertFalse((boolean)ByteBufferUtils.equals((ByteBuffer)bb, (int)0, (int)0, (ByteBuffer)HConstants.EMPTY_BYTE_BUFFER, (int)0, (int)a.length));
        Assertions.assertTrue((boolean)ByteBufferUtils.equals((ByteBuffer)bb, (int)0, (int)a.length, (ByteBuffer)bb, (int)0, (int)a.length));
        Assertions.assertTrue((boolean)ByteBufferUtils.equals((ByteBuffer)HConstants.EMPTY_BYTE_BUFFER, (int)0, (int)0, (byte[])HConstants.EMPTY_BYTE_ARRAY, (int)0, (int)0));
        Assertions.assertFalse((boolean)ByteBufferUtils.equals((ByteBuffer)HConstants.EMPTY_BYTE_BUFFER, (int)0, (int)0, (byte[])a, (int)0, (int)a.length));
        Assertions.assertFalse((boolean)ByteBufferUtils.equals((ByteBuffer)bb, (int)0, (int)a.length, (byte[])HConstants.EMPTY_BYTE_ARRAY, (int)0, (int)0));
        Assertions.assertTrue((boolean)ByteBufferUtils.equals((ByteBuffer)bb, (int)0, (int)a.length, (byte[])a, (int)0, (int)a.length));
    }

    @Test
    public void testFindCommonPrefix() {
        ByteBuffer bb1 = ByteBuffer.allocate(135);
        ByteBuffer bb2 = ByteBuffer.allocate(135);
        ByteBuffer bb3 = ByteBuffer.allocateDirect(135);
        byte[] b = new byte[71];
        ByteBufferUtilsTestBase.fillBB(bb1, (byte)5);
        ByteBufferUtilsTestBase.fillBB(bb2, (byte)5);
        ByteBufferUtilsTestBase.fillBB(bb3, (byte)5);
        ByteBufferUtilsTestBase.fillArray(b, (byte)5);
        Assertions.assertEquals((int)135, (int)ByteBufferUtils.findCommonPrefix((ByteBuffer)bb1, (int)0, (int)bb1.remaining(), (ByteBuffer)bb2, (int)0, (int)bb2.remaining()));
        Assertions.assertEquals((int)71, (int)ByteBufferUtils.findCommonPrefix((ByteBuffer)bb1, (int)0, (int)bb1.remaining(), (byte[])b, (int)0, (int)b.length));
        Assertions.assertEquals((int)135, (int)ByteBufferUtils.findCommonPrefix((ByteBuffer)bb1, (int)0, (int)bb1.remaining(), (ByteBuffer)bb3, (int)0, (int)bb3.remaining()));
        Assertions.assertEquals((int)71, (int)ByteBufferUtils.findCommonPrefix((ByteBuffer)bb3, (int)0, (int)bb3.remaining(), (byte[])b, (int)0, (int)b.length));
        b[13] = 9;
        Assertions.assertEquals((int)13, (int)ByteBufferUtils.findCommonPrefix((ByteBuffer)bb1, (int)0, (int)bb1.remaining(), (byte[])b, (int)0, (int)b.length));
        bb2.put(134, (byte)6);
        Assertions.assertEquals((int)134, (int)ByteBufferUtils.findCommonPrefix((ByteBuffer)bb1, (int)0, (int)bb1.remaining(), (ByteBuffer)bb2, (int)0, (int)bb2.remaining()));
        bb2.put(6, (byte)4);
        Assertions.assertEquals((int)6, (int)ByteBufferUtils.findCommonPrefix((ByteBuffer)bb1, (int)0, (int)bb1.remaining(), (ByteBuffer)bb2, (int)0, (int)bb2.remaining()));
    }

    private static void testCompressedInt(int value) throws IOException {
        ByteArrayOutputStream bos = new ByteArrayOutputStream();
        ByteBufferUtils.putCompressedInt((OutputStream)bos, (int)value);
        ByteArrayInputStream bis = new ByteArrayInputStream(bos.toByteArray());
        int parsedValue = ByteBufferUtils.readCompressedInt((InputStream)bis);
        Assertions.assertEquals((int)value, (int)parsedValue);
    }

    private static void testPutInt(int value) {
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        try {
            ByteBufferUtils.putInt((OutputStream)baos, (int)value);
        }
        catch (IOException e) {
            throw new RuntimeException("Bug in putIn()", e);
        }
        ByteArrayInputStream bais = new ByteArrayInputStream(baos.toByteArray());
        DataInputStream dis = new DataInputStream(bais);
        try {
            Assertions.assertEquals((int)dis.readInt(), (int)value);
        }
        catch (IOException e) {
            throw new RuntimeException("Bug in test!", e);
        }
    }

    private static void fillBB(ByteBuffer bb, byte b) {
        for (int i = bb.position(); i < bb.limit(); ++i) {
            bb.put(i, b);
        }
    }

    private static void fillArray(byte[] bb, byte b) {
        for (int i = 0; i < bb.length; ++i) {
            bb[i] = b;
        }
    }
}

