/*
 * Decompiled with CFR 0.152.
 */
package org.apache.amoro.server.catalog;

import java.time.Duration;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ExecutionException;
import java.util.stream.Collectors;
import org.apache.amoro.AmoroTable;
import org.apache.amoro.api.CatalogMeta;
import org.apache.amoro.config.Configurations;
import org.apache.amoro.exception.AlreadyExistsException;
import org.apache.amoro.exception.IllegalMetadataException;
import org.apache.amoro.exception.ObjectNotExistsException;
import org.apache.amoro.server.AmoroManagementConf;
import org.apache.amoro.server.catalog.CatalogBuilder;
import org.apache.amoro.server.catalog.CatalogManager;
import org.apache.amoro.server.catalog.InternalCatalog;
import org.apache.amoro.server.catalog.ServerCatalog;
import org.apache.amoro.server.dashboard.utils.AmsUtil;
import org.apache.amoro.server.persistence.PersistentBase;
import org.apache.amoro.server.persistence.mapper.CatalogMetaMapper;
import org.apache.amoro.shade.guava32.com.google.common.annotations.VisibleForTesting;
import org.apache.amoro.shade.guava32.com.google.common.cache.CacheBuilder;
import org.apache.amoro.shade.guava32.com.google.common.cache.CacheLoader;
import org.apache.amoro.shade.guava32.com.google.common.cache.LoadingCache;
import org.apache.amoro.shade.guava32.com.google.common.collect.Maps;
import org.apache.amoro.table.TableIdentifier;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DefaultCatalogManager
extends PersistentBase
implements CatalogManager {
    private static final Logger LOG = LoggerFactory.getLogger(DefaultCatalogManager.class);
    protected final Configurations serverConfiguration;
    private final LoadingCache<String, Optional<CatalogMeta>> metaCache;
    private final Map<String, ServerCatalog> serverCatalogMap = Maps.newConcurrentMap();

    public DefaultCatalogManager(Configurations serverConfiguration) {
        this.serverConfiguration = serverConfiguration;
        Duration cacheTtl = (Duration)serverConfiguration.get(AmoroManagementConf.CATALOG_META_CACHE_EXPIRATION_INTERVAL);
        this.metaCache = CacheBuilder.newBuilder().maximumSize(100L).expireAfterWrite(cacheTtl).build((CacheLoader)new CacheLoader<String, Optional<CatalogMeta>>(){

            @NotNull
            public Optional<CatalogMeta> load(@NotNull String key) throws Exception {
                return Optional.ofNullable(DefaultCatalogManager.this.getAs(CatalogMetaMapper.class, mapper -> mapper.getCatalog(key)));
            }
        });
        this.listCatalogMetas().forEach(c -> {
            ServerCatalog serverCatalog = CatalogBuilder.buildServerCatalog(c, serverConfiguration);
            this.serverCatalogMap.put(c.getCatalogName(), serverCatalog);
            this.metaCache.put((Object)c.getCatalogName(), Optional.of(c));
            LOG.info("Load catalog {}, type:{}", (Object)c.getCatalogName(), (Object)c.getCatalogType());
        });
        LOG.info("DefaultCatalogManager initialized, total catalogs: {}", (Object)this.serverCatalogMap.size());
    }

    @Override
    public List<CatalogMeta> listCatalogMetas() {
        return this.getAs(CatalogMetaMapper.class, CatalogMetaMapper::getCatalogs).stream().peek(c -> this.metaCache.put((Object)c.getCatalogName(), Optional.of(c))).collect(Collectors.toList());
    }

    @Override
    public CatalogMeta getCatalogMeta(String catalogName) {
        return this.getCatalogMetaOptional(catalogName).orElseThrow(() -> new ObjectNotExistsException("Catalog " + catalogName)).deepCopy();
    }

    private Optional<CatalogMeta> getCatalogMetaOptional(String catalogName) {
        try {
            return (Optional)this.metaCache.get((Object)catalogName);
        }
        catch (ExecutionException e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public boolean catalogExist(String catalogName) {
        return this.getCatalogMetaOptional(catalogName).isPresent();
    }

    @Override
    public ServerCatalog getServerCatalog(String catalogName) {
        Optional<CatalogMeta> catalogMeta = this.getCatalogMetaOptional(catalogName);
        if (!catalogMeta.isPresent()) {
            this.disposeCatalog(catalogName);
            throw new ObjectNotExistsException("Catalog " + catalogName);
        }
        ServerCatalog serverCatalog = this.serverCatalogMap.computeIfAbsent(catalogName, n -> CatalogBuilder.buildServerCatalog((CatalogMeta)catalogMeta.get(), this.serverConfiguration));
        serverCatalog.reload(catalogMeta.get());
        return serverCatalog;
    }

    @VisibleForTesting
    public void setServerCatalog(ServerCatalog catalog) {
        this.serverCatalogMap.put(catalog.name(), catalog);
    }

    @Override
    public InternalCatalog getInternalCatalog(String catalogName) {
        ServerCatalog serverCatalog = this.getServerCatalog(catalogName);
        if (serverCatalog == null) {
            throw new ObjectNotExistsException("Catalog " + catalogName);
        }
        if (serverCatalog.isInternal()) {
            return (InternalCatalog)serverCatalog;
        }
        throw new ObjectNotExistsException("Catalog " + catalogName + " is not internal catalog");
    }

    @Override
    public List<ServerCatalog> getServerCatalogs() {
        return this.listCatalogMetas().stream().map(c -> {
            ServerCatalog serverCatalog = this.serverCatalogMap.computeIfAbsent(c.getCatalogName(), n -> CatalogBuilder.buildServerCatalog(c, this.serverConfiguration));
            serverCatalog.reload((CatalogMeta)c);
            return serverCatalog;
        }).collect(Collectors.toList());
    }

    @Override
    public void createCatalog(CatalogMeta catalogMeta) {
        if (this.catalogExist(catalogMeta.getCatalogName())) {
            throw new AlreadyExistsException("Catalog " + catalogMeta.getCatalogName());
        }
        this.fillCatalogProperties(catalogMeta);
        ServerCatalog catalog = CatalogBuilder.buildServerCatalog(catalogMeta, this.serverConfiguration);
        this.doAs(CatalogMetaMapper.class, mapper -> mapper.insertCatalog(catalog.getMetadata()));
        this.disposeCatalog(catalogMeta.getCatalogName());
        this.serverCatalogMap.put(catalogMeta.getCatalogName(), catalog);
        LOG.info("Create catalog {}, type:{}", (Object)catalogMeta.getCatalogName(), (Object)catalogMeta.getCatalogType());
    }

    private void fillCatalogProperties(CatalogMeta catalogMeta) {
        String type = catalogMeta.getCatalogType();
        if ("hive".equals(type)) {
            String amsUri = AmsUtil.getAMSThriftAddress(this.serverConfiguration, "TableMetastore");
            catalogMeta.getCatalogProperties().put("ams.uri", amsUri);
            catalogMeta.getCatalogProperties().putIfAbsent("clients", String.valueOf(20));
        }
    }

    @Override
    public void dropCatalog(String catalogName) {
        this.doAs(CatalogMetaMapper.class, mapper -> {
            CatalogMeta meta = mapper.getCatalog(catalogName);
            if (DefaultCatalogManager.isInternal(meta)) {
                int dbCount = mapper.selectDatabaseCount(catalogName);
                int tblCount = mapper.selectTableCount(catalogName);
                if (dbCount > 0 || tblCount > 0) {
                    throw new IllegalMetadataException("Cannot drop internal catalog with databases or tables");
                }
            }
            mapper.deleteCatalog(catalogName);
            this.metaCache.invalidate((Object)catalogName);
        });
        this.disposeCatalog(catalogName);
    }

    @Override
    public void updateCatalog(CatalogMeta catalogMeta) {
        ServerCatalog catalog = this.getServerCatalog(catalogMeta.getCatalogName());
        this.validateCatalogUpdate(catalog.getMetadata(), catalogMeta);
        this.metaCache.invalidate((Object)catalogMeta.getCatalogName());
        catalog.updateMetadata(catalogMeta);
        LOG.info("Update catalog metadata: {}", (Object)catalogMeta.getCatalogName());
    }

    @Override
    public AmoroTable<?> loadTable(TableIdentifier identifier) {
        ServerCatalog serverCatalog = this.getServerCatalog(identifier.getCatalog());
        return serverCatalog.loadTable(identifier.getDatabase(), identifier.getTableName());
    }

    private void validateCatalogUpdate(CatalogMeta oldMeta, CatalogMeta newMeta) {
        if (!oldMeta.getCatalogType().equals(newMeta.getCatalogType())) {
            throw new IllegalMetadataException("Cannot update catalog type");
        }
    }

    private void disposeCatalog(String name) {
        this.serverCatalogMap.computeIfPresent(name, (n, c) -> {
            LOG.info("Dispose catalog: {}", n);
            c.dispose();
            return null;
        });
        this.metaCache.invalidate((Object)name);
    }

    private static boolean isInternal(CatalogMeta meta) {
        return "ams".equalsIgnoreCase(meta.getCatalogType());
    }
}

